import os
import bz2
import re
import pandas as pd
import pytz
from datetime import datetime, timedelta, time
import great_circle_calculator.great_circle_calculator as gcc
import folium
from tqdm import tqdm



# ---------- FUNZIONI DI SUPPORTO ----------

def carica_bz2_in_cache(slaeq_dir):
    bz2_cache = {}
    for postazione in os.listdir(slaeq_dir):
        postazione_dir = os.path.join(slaeq_dir, postazione)
        if not os.path.isdir(postazione_dir):
            continue
        for filename in os.listdir(postazione_dir):
            if filename.endswith(".slaeq.bz2"):
                key = filename.replace(".slaeq.bz2", "")
                filepath = os.path.join(postazione_dir, filename)
                with bz2.open(filepath, 'rt') as f:
                    df = pd.read_csv(f, sep=r'\s+', header=None, names=['giorno', 'mese', 'anno', 'orario', 'valore'])
                    if df.empty or df.shape[1] < 5:
                        continue
                    df = df[df["orario"].astype(str).str.match(r"^\d{2}:\d{2}:\d{2}$")]
                    df['timestamp'] = pd.to_datetime(df[['anno', 'mese', 'giorno']].astype(str).agg('-'.join, axis=1) + ' ' + df['orario'])
                    df.set_index('timestamp', inplace=True)
                    bz2_cache[key] = df
    return bz2_cache


def leggi_valore_decibel(data_trasformata, timestamp_radar, postazione_vicina, bz2_cache):
    key = f"{postazione_vicina}_{data_trasformata}"
    df = bz2_cache.get(key)
    if df is None:
        return "battuta non disponibile"
    timestamp = pd.to_datetime(timestamp_radar)
    if timestamp in df.index:
        return float(df.at[timestamp, 'valore'])
    else:
        return "battuta non disponibile"


def carica_dati_meteo(path):
    df = pd.read_csv(path)
    df['valid'] = pd.to_datetime(df['valid'])
    return df.set_index('valid')


def carica_dati_aircraft(path):
    df = pd.read_csv(path, encoding='cp1252')
    df['DATA'] = pd.to_datetime(df['DATA'], errors='coerce').dt.date
    return df


def carica_dati_matched(path):
    df = pd.read_csv(path, encoding='cp1252')
    return df


def carica_medie(folder_path):
    medie = {}
    with open(folder_path) as f:
        current_centralina = ''
        for line in f:
            line = line.strip()
            if not line:
                continue
            if line.startswith("Riepilogo"):
                current_centralina = line.split()[-1]
                medie[current_centralina] = {}
                continue
            if line.startswith("Data") or line.startswith("Notte"):
                try:
                    label, valore = line.split(":", 1)
                    valore = valore.strip().lower()
                    if label.startswith("Data"):
                        data = label.split()[1]
                    elif label.startswith("Notte"):
                        data = label.split()[1].split("-")[0]  # prendi solo la prima data
                    else:
                        continue
                    # Valore assente
                    if "file vuoto" in valore or "nessun dato" in valore:
                        medie[current_centralina][data] = None
                    else:
                        valore_db = float(valore.replace("db", "").strip())
                        medie[current_centralina][data] = valore_db
                except ValueError:
                    continue
    return medie


def round_to_nearest_20_or_50(dt):
    if dt.minute <= 4:
        return dt.replace(minute=50, second=0, microsecond=0) - timedelta(hours=1)
    elif dt.minute <= 34:
        return dt.replace(minute=20, second=0, microsecond=0)
    else:
        return dt.replace(minute=50, second=0, microsecond=0)



# ---------- SETUP PERCORSI ----------
base_dir = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Tutte le giornate\Risultati plot con radar_read'
slaeq_dir = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\slaeq'
meteo_path = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Callsign - centralina - scostamento - vento - motore\RIDOTTO LIMF Iowa Environmental Mesonet.csv'
aircraft_path = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Tutte le giornate\aircraft_detail_2023 - 2024.csv'
matched_path = r"C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Tutte le giornate\matched_missing_callsign 2023-2024 - senza TFS.csv"
folder_path_diurna = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Calcolo medie\Calcolo media diurna.txt'
folder_path_notturna = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Calcolo medie\Calcolo media notturna.txt'
output_dir = r'C:\Users\silvia.convertini\OneDrive - Torino Airport\Desktop\Visual Studio Code\Tutte le giornate\Risultati MATCHED Callsign - centralina - scostamento - vento - motore - angolo'



# ---------- PRE-CARICAMENTO DATI ----------
bz2_cache = carica_bz2_in_cache(slaeq_dir)
df_meteo = carica_dati_meteo(meteo_path)
df_aircraft = carica_dati_aircraft(aircraft_path)
df_matched = carica_dati_matched(matched_path)
medie_diurna = carica_medie(folder_path_diurna)
medie_notturna = carica_medie(folder_path_notturna)
print("Caricamento completato.")



# ---------- POSTAZIONI ----------
postazioni = {
    "LIMF01": (45.22434804511844, 7.659483164436543),
    "LIMF02": (45.23125911377918, 7.6543043633872605),
    "LIMF03": (45.233018221859346, 7.640868635320528),
    "LIMF04": (45.17795651909228, 7.641713999737966),
    "LIMF05": (45.18024207852103, 7.648466270903088),
    "LIMF07": (45.2027055381157, 7.654697541282107),
    "LIMF10": (45.21354388314895, 7.638394710603071),
}



# ---------- LOOP ANALISI FILE ----------
for filename in tqdm(os.listdir(base_dir), desc="Elaborazione file radar"):
    file_path = os.path.join(base_dir, filename)
    output_path = os.path.join(output_dir, f"DATI {filename}")

    with open(file_path, "r") as file:
        next(file)
        lines = file.readlines()

    risultati = []
    for line in lines:
        parts = re.split(r'[;, ]+', line.strip())
        if len(parts) < 11:
            continue

        flight = {
            "callsign": parts[0],
            "model": parts[1],
            "from": parts[2],
            "to": parts[3],
            "timestamp": parts[4] + ' ' + parts[5],
            "lat": float(parts[6]),
            "lon": float(parts[7]),
            "quota": float(parts[8]),
            "postazione": parts[9],
            "distanza": float(parts[10]),
        }


        utc_naive = datetime.strptime(flight["timestamp"], "%Y-%m-%d %H:%M:%S")
        utc_aware = utc_naive.replace(tzinfo=pytz.utc)
        local_dt = utc_aware.astimezone(pytz.timezone('Europe/Rome'))
        ora_locale = local_dt.strftime("%H:%M:%S")
        data_locale = local_dt.strftime("%Y-%m-%d")

        data_trasformata = local_dt.strftime("%y%m%d")
        flight["timestamp_locale"] = f"{data_locale} {ora_locale}"
        decibel = leggi_valore_decibel(data_trasformata, flight["timestamp_locale"], flight["postazione"], bz2_cache)

        if isinstance(decibel, str):
            continue  # skippa la riga se valore dB centralina non disponibile


        rounded_dt = round_to_nearest_20_or_50(utc_naive)
        meteo_ts = pd.to_datetime(rounded_dt.strftime("%Y-%m-%d %H:%M:%S"))
        
        if meteo_ts in df_meteo.index:
            valori_meteo = df_meteo.loc[meteo_ts]
        else:
            valori_meteo is None


        lat_centralina, lon_centralina = postazioni[flight["postazione"]]
        bearing = 90 - gcc.bearing_at_p1((flight["lat"], flight["lon"]), (lat_centralina, lon_centralina))

        wind_dir = valori_meteo["drct"]
        if wind_dir != "M":
            angolo_relativo = abs((bearing % 360) - float(wind_dir))
            if angolo_relativo <= 45 or angolo_relativo >= 315:
                effetto = 2  # "Sopravento"
            elif 135 <= angolo_relativo <= 225:
                effetto = 1  # "Sottovento"
            else:
                effetto = 0  # "Laterale"
        else:
            angolo_relativo = "NULL"
            effetto = "NULL"


        callsign_TFS = ["ITFSA", "ITFSB", "ITFSC", "ITFSD", "ICGAD", "ILDAP", "ISMBR"]
        if flight["callsign"] in callsign_TFS:
            dettagli = df_aircraft[
                (df_aircraft['DATA'] == local_dt.date()) & (df_aircraft['AIRCRAFT_MATRICULA'] == flight["callsign"])
            ]
        else:
            dettagli = df_aircraft[
                (df_aircraft['DATA'] == local_dt.date()) & (df_aircraft['CALL_SIGN'] == flight["callsign"])
            ]

        # se questo controllo restituisce NULL allora cercare nel file matched callsign solo il callsign
        if dettagli.empty:
            dettagli = df_matched[df_matched['CALL_SIGN'] == flight["callsign"]]

        # se anche questo controllo restituisce NULL allora cercare nel file aircraft detail 2023-2024 solo il callsign con arrivo o partenza
        if dettagli.empty:
            if flight["from"] == "LIMF":
                dettagli = df_aircraft[(df_aircraft['CALL_SIGN'] == flight["callsign"]) & (df_aircraft['NATURE_CODE'] == "P")]
            elif flight["to"] == "LIMF":
                dettagli = df_aircraft[(df_aircraft['CALL_SIGN'] == flight["callsign"]) & (df_aircraft['NATURE_CODE'] == "A")]


        if not dettagli.empty:
            r = dettagli.iloc[0]
            AIRCRAFT_ICAO_MODEL = r["AIRCRAFT_ICAO_MODEL"]
            AIRCRAFT_MATRICULA = r["AIRCRAFT_MATRICULA"]
            NOISE = r["NOISE"]
            RWY = r["RWY"]
            NATURE_CODE = r["NATURE_CODE"]
        else:
            AIRCRAFT_ICAO_MODEL = AIRCRAFT_MATRICULA = NOISE = RWY = NATURE_CODE = "NULL"


        ora_locale_time = local_dt.time()
        if time(6, 0, 0) <= ora_locale_time <= time(22, 59, 59):
            fascia = 'diurna'
            data_per_media = local_dt.date()
        else:
            fascia = 'notturna'
            data_per_media = (local_dt - timedelta(days=1)).date() if ora_locale_time < time(6, 0, 0) else local_dt.date()

        data_media = data_per_media.strftime("%y%m%d")
        media = (medie_diurna if fascia == 'diurna' else medie_notturna).get(flight["postazione"], {}).get(data_media)

        if media is not None:
            diff = decibel - media
        else:
            diff = "NULL"


        risultati.append([
            flight["callsign"], flight["from"], flight["to"], flight["lat"], flight["lon"], flight["quota"], flight["postazione"],
            flight["distanza"], data_locale, ora_locale, decibel, media, diff,
            valori_meteo["tmpc"], valori_meteo["dwpc"], valori_meteo["relh"],
            wind_dir, valori_meteo["vento_ms"], bearing, angolo_relativo, effetto,
            AIRCRAFT_ICAO_MODEL, AIRCRAFT_MATRICULA, NOISE, RWY, NATURE_CODE
        ])


    df = pd.DataFrame(risultati, columns=[
        "Callsign", "From", "To", "Lat", "Lon", "Quota", "Centralina", "Distanza", "Data_locale", "Orario_locale", "dB_rilevato", "dB_medio",
        "Scostamento", "Air_Temperature", "Dew_Point_Temperature", "Relative_Humidity", "Wind_Direction",
        "Wind_Speed", "Bearing", "Angolo_relativo", "Effetto_vento", "AIRCRAFT_ICAO_MODEL", "AIRCRAFT_MATRICULA",
        "NOISE", "RWY", "NATURE_CODE"
    ])
    df.to_csv(output_path, index=False)